using System;
using System.IO;
using DynaPDF;

namespace barcodes
{
	class Barcodes
	{
      public struct TTestBarcode
      {
         public TPDFBarcodeType     BarcodeType;
         public String              BarcodeName;
         public TPDFBarcodeDataType DataType;
         public String              Data;
         public String              Primary;

         public TTestBarcode(TPDFBarcodeType BcdType, String Name, TPDFBarcodeDataType DtaType, String Dta, String Prmy)
         {
            BarcodeType = BcdType;
            BarcodeName = Name;
            DataType    = DtaType;
            Data        = Dta;
            Primary     = Prmy;
         }
      }
      public static TTestBarcode[] TEST_CODES =
      {
         new TTestBarcode(TPDFBarcodeType.bctAustraliaPost, "Australia Post", TPDFBarcodeDataType.bcdtBinary, "12345678", ""),
         new TTestBarcode(TPDFBarcodeType.bctAustraliaRedir, "Australia Redirect Code", TPDFBarcodeDataType.bcdtBinary, "12345678", ""),
         new TTestBarcode(TPDFBarcodeType.bctAustraliaReply, "Australia Reply-Paid", TPDFBarcodeDataType.bcdtBinary, "12345678", ""),
         new TTestBarcode(TPDFBarcodeType.bctAustraliaRout, "Australia Routing Code", TPDFBarcodeDataType.bcdtBinary, "12345678", ""),
         new TTestBarcode(TPDFBarcodeType.bctAztec, "Aztec binary mode", TPDFBarcodeDataType.bcdtBinary, "123456789012", ""),
         new TTestBarcode(TPDFBarcodeType.bctAztec, "Aztec GS1 Mode", TPDFBarcodeDataType.bcdtGS1Mode, "[01]03453120000011[17]120508[10]ABCD1234[410]9501101020917", ""),
         new TTestBarcode(TPDFBarcodeType.bctAztecRunes, "Aztec Runes", TPDFBarcodeDataType.bcdtBinary, "123", ""),
         new TTestBarcode(TPDFBarcodeType.bctC2Of5IATA, "Code 2 of 5 IATA", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctC2Of5Industrial, "Code 2 of 5 Industrial", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctC2Of5Interleaved, "Code 2 of 5 Interleaved", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctC2Of5Logic, "Code 2 of 5 Data Logic", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctC2Of5Matrix, "Code 2 of 5 Matrix", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctChannelCode, "Channel Code", TPDFBarcodeDataType.bcdtBinary, "1234567", ""),
         new TTestBarcode(TPDFBarcodeType.bctCodabar, "Codabar", TPDFBarcodeDataType.bcdtBinary, "A123456789B", ""),
         new TTestBarcode(TPDFBarcodeType.bctCodablockF, "Codablock-F", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdefghijklmnopqrstuvwxyz", ""),
         new TTestBarcode(TPDFBarcodeType.bctCode11, "Code 11", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctCode128, "Code 128", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctCode128B, "Code 128", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctCode16K, "Code 16K binary mode", TPDFBarcodeDataType.bcdtBinary, "[90]A1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctCode16K, "Code 16K GS1 mode", TPDFBarcodeDataType.bcdtGS1Mode, "[90]A1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctCode32, "Code 32", TPDFBarcodeDataType.bcdtBinary, "12345678", ""),
         new TTestBarcode(TPDFBarcodeType.bctCode39, "Code 39", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctCode49, "Code 49", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctCode93, "Code 93", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctCodeOne, "Code One", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctDAFT, "DAFT Code", TPDFBarcodeDataType.bcdtBinary, "aftdaftdftaft", ""),
         new TTestBarcode(TPDFBarcodeType.bctDataBarOmniTrunc, "GS1 DataBar Omnidirectional", TPDFBarcodeDataType.bcdtBinary, "0123456789012", ""),
         new TTestBarcode(TPDFBarcodeType.bctDataBarExpStacked, "GS1 DataBar Stacked", TPDFBarcodeDataType.bcdtBinary, "[90]1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctDataBarExpanded, "GS1 DataBar Expanded", TPDFBarcodeDataType.bcdtBinary, "[90]1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctDataBarLimited, "GS1 DataBar Limited", TPDFBarcodeDataType.bcdtBinary, "0123456789012", ""),
         new TTestBarcode(TPDFBarcodeType.bctDataBarStacked, "GS1 DataBar Stacked", TPDFBarcodeDataType.bcdtBinary, "0123456789012", ""),
         new TTestBarcode(TPDFBarcodeType.bctDataBarStackedO, "GS1 DataBar Stacked Omni", TPDFBarcodeDataType.bcdtBinary, "0123456789012", ""),
         new TTestBarcode(TPDFBarcodeType.bctDataMatrix, "Data Matrix ISO 16022", TPDFBarcodeDataType.bcdtBinary, "0123456789012", ""),
         new TTestBarcode(TPDFBarcodeType.bctDotCode, "DotCode", TPDFBarcodeDataType.bcdtBinary, "0123456789012", ""),
         new TTestBarcode(TPDFBarcodeType.bctDPD, "DPD Code", TPDFBarcodeDataType.bcdtBinary, "1234567890123456789012345678", ""),
         new TTestBarcode(TPDFBarcodeType.bctDPIdentCode, "Deutsche Post Identcode", TPDFBarcodeDataType.bcdtBinary, "12345678901", ""),
         new TTestBarcode(TPDFBarcodeType.bctDPLeitcode, "Deutsche Post Leitcode", TPDFBarcodeDataType.bcdtBinary, "1234567890123", ""),
         new TTestBarcode(TPDFBarcodeType.bctEAN128, "EAN 128", TPDFBarcodeDataType.bcdtBinary, "[90]0101234567890128TEC-IT", ""),
         new TTestBarcode(TPDFBarcodeType.bctEAN128_CC, "EAN 128 Composite Code", TPDFBarcodeDataType.bcdtBinary, "[10]1234-1234", "[90]123456"),
         new TTestBarcode(TPDFBarcodeType.bctEAN14, "EAN 14", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctEANX, "EAN X", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctEANX_CC, "EAN Composite Symbol", TPDFBarcodeDataType.bcdtBinary, "[90]12341234", "12345678"),
         new TTestBarcode(TPDFBarcodeType.bctEANXCheck, "EAN + Check Digit", TPDFBarcodeDataType.bcdtBinary, "12345", ""),
         new TTestBarcode(TPDFBarcodeType.bctExtCode39, "Ext. Code 3 of 9 (Code 39+)", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctFIM, "FIM", TPDFBarcodeDataType.bcdtBinary, "d", ""),
         new TTestBarcode(TPDFBarcodeType.bctFlattermarken, "Flattermarken", TPDFBarcodeDataType.bcdtBinary, "11111111111111", ""),
         new TTestBarcode(TPDFBarcodeType.bctHIBC_Aztec, "HIBC Aztec Code", TPDFBarcodeDataType.bcdtBinary, "123456789012", ""),
         new TTestBarcode(TPDFBarcodeType.bctHIBC_CodablockF, "HIBC Codablock-F", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdefghijklmnopqrstuvwxyz", ""),
         new TTestBarcode(TPDFBarcodeType.bctHIBC_Code128, "HIBC Code 128", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctHIBC_Code39, "HIBC Code 39", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctHIBC_DataMatrix, "HIBC Data Matrix", TPDFBarcodeDataType.bcdtBinary, "0123456789012", ""),
         new TTestBarcode(TPDFBarcodeType.bctHIBC_MicroPDF417, "HIBC Micro PDF417", TPDFBarcodeDataType.bcdtBinary, "01234567890abcde", ""),
         new TTestBarcode(TPDFBarcodeType.bctHIBC_PDF417, "HIBC PDF417", TPDFBarcodeDataType.bcdtBinary, "01234567890abcde", ""),
         new TTestBarcode(TPDFBarcodeType.bctHIBC_QR, "HIBC QR Code", TPDFBarcodeDataType.bcdtBinary, "01234567890abcde", ""),
         new TTestBarcode(TPDFBarcodeType.bctISBNX, "ISBN (EAN-13 with validation)", TPDFBarcodeDataType.bcdtBinary, "0123456789", ""),
         new TTestBarcode(TPDFBarcodeType.bctITF14, "ITF-14", TPDFBarcodeDataType.bcdtBinary, "0123456789", ""),
         new TTestBarcode(TPDFBarcodeType.bctJapanPost, "Japanese Postal Code", TPDFBarcodeDataType.bcdtBinary, "0123456789", ""),
         new TTestBarcode(TPDFBarcodeType.bctKIX, "Dutch Post KIX Code", TPDFBarcodeDataType.bcdtBinary, "0123456789", ""),
         new TTestBarcode(TPDFBarcodeType.bctKoreaPost, "Korea Post", TPDFBarcodeDataType.bcdtBinary, "123456", ""),
         new TTestBarcode(TPDFBarcodeType.bctLOGMARS, "LOGMARS", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdef", ""),
         new TTestBarcode(TPDFBarcodeType.bctMailmark, "Royal Mail 4-State Mailmark", TPDFBarcodeDataType.bcdtBinary, "11210012341234567AB19XY1A", ""),
         new TTestBarcode(TPDFBarcodeType.bctMaxicode, "Maxicode", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdef", ""),
         new TTestBarcode(TPDFBarcodeType.bctMicroPDF417, "Micro PDF417", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdef", ""),
         new TTestBarcode(TPDFBarcodeType.bctMicroQR, "Micro QR Code", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdef", ""),
         new TTestBarcode(TPDFBarcodeType.bctMSIPlessey, "MSI Plessey", TPDFBarcodeDataType.bcdtBinary, "12345678901", ""),
         new TTestBarcode(TPDFBarcodeType.bctNVE18, "NVE-18", TPDFBarcodeDataType.bcdtBinary, "1234567890123456", ""),
         new TTestBarcode(TPDFBarcodeType.bctPDF417, "PDF417", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdef", ""),
         new TTestBarcode(TPDFBarcodeType.bctPDF417Truncated, "PDF417 Truncated", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdef", ""),
         new TTestBarcode(TPDFBarcodeType.bctPharmaOneTrack, "Pharmacode One-Track", TPDFBarcodeDataType.bcdtBinary, "123456", ""),
         new TTestBarcode(TPDFBarcodeType.bctPharmaTwoTrack, "Pharmacode Two-Track", TPDFBarcodeDataType.bcdtBinary, "123456", ""),
         new TTestBarcode(TPDFBarcodeType.bctPLANET, "PLANET", TPDFBarcodeDataType.bcdtBinary, "12345678901", ""),
         new TTestBarcode(TPDFBarcodeType.bctPlessey, "Plessey", TPDFBarcodeDataType.bcdtBinary, "12345678901", ""),
         new TTestBarcode(TPDFBarcodeType.bctPostNet, "PostNet", TPDFBarcodeDataType.bcdtBinary, "12345678901", ""),
         new TTestBarcode(TPDFBarcodeType.bctPZN, "PZN", TPDFBarcodeDataType.bcdtBinary, "1234567", ""),
         new TTestBarcode(TPDFBarcodeType.bctQRCode, "QR Code", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdef", ""),
         new TTestBarcode(TPDFBarcodeType.bctRMQR, "Rect. Micro QR Code (rMQR)", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdef", ""),
         new TTestBarcode(TPDFBarcodeType.bctRoyalMail4State, "Royal Mail 4 State (RM4SCC)", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdef", ""),
         new TTestBarcode(TPDFBarcodeType.bctRSS_EXP_CC, "CS GS1 DataBar Ext. component", TPDFBarcodeDataType.bcdtBinary, "[90]12341234", "[10]12345678"),
         new TTestBarcode(TPDFBarcodeType.bctRSS_EXPSTACK_CC, "CS GS1 DataBar Exp. Stacked", TPDFBarcodeDataType.bcdtBinary, "[90]12341234", "[10]12345678"),
         new TTestBarcode(TPDFBarcodeType.bctRSS_LTD_CC, "CS GS1 DataBar Limited", TPDFBarcodeDataType.bcdtBinary, "[90]12341234", "1234567"),
         new TTestBarcode(TPDFBarcodeType.bctRSS14_CC, "CS GS1 DataBar-14 Linear", TPDFBarcodeDataType.bcdtBinary, "[90]12341234", "1234567"),
         new TTestBarcode(TPDFBarcodeType.bctRSS14Stacked_CC, "CS GS1 DataBar-14 Stacked", TPDFBarcodeDataType.bcdtBinary, "[90]12341234", "1234567"),
         new TTestBarcode(TPDFBarcodeType.bctRSS14StackOMNI_CC, "CS GS1 DataBar-14 Stacked Omni", TPDFBarcodeDataType.bcdtBinary, "[90]12341234", "1234567"),
         new TTestBarcode(TPDFBarcodeType.bctTelepen, "Telepen Alpha", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdef", ""),
         new TTestBarcode(TPDFBarcodeType.bctUltracode, "Ultracode", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdef", ""),
         new TTestBarcode(TPDFBarcodeType.bctUPCA, "UPC A", TPDFBarcodeDataType.bcdtBinary, "1234567890", ""),
         new TTestBarcode(TPDFBarcodeType.bctUPCA_CC, "CS UPC A linear", TPDFBarcodeDataType.bcdtBinary, "[90]12341234", "1234567"),
         new TTestBarcode(TPDFBarcodeType.bctUPCACheckDigit, "UPC A + Check Digit", TPDFBarcodeDataType.bcdtBinary, "12345678905", ""),
         new TTestBarcode(TPDFBarcodeType.bctUPCE, "UCP E", TPDFBarcodeDataType.bcdtBinary, "1234567", ""),
         new TTestBarcode(TPDFBarcodeType.bctUPCE_CC, "CS UPC E linear", TPDFBarcodeDataType.bcdtBinary, "[90]12341234", "1234567"),
         new TTestBarcode(TPDFBarcodeType.bctUPCECheckDigit, "UCP E + Check Digit", TPDFBarcodeDataType.bcdtBinary, "12345670", ""),
         new TTestBarcode(TPDFBarcodeType.bctUPNQR, "UPNQR (Univ. Placilni Nalog QR)", TPDFBarcodeDataType.bcdtBinary, "1234567890abcdef", ""),
         new TTestBarcode(TPDFBarcodeType.bctUSPSOneCode, "USPS OneCode", TPDFBarcodeDataType.bcdtBinary, "01234567094987654321", ""),
         new TTestBarcode(TPDFBarcodeType.bctVIN, "Vehicle Ident Number (USA)", TPDFBarcodeDataType.bcdtBinary, "01234567094987654", "")
      };
	
	
      /*
         Note that the dynapdf.dll must be copied into the output directory or into a
         Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
      */

      // Error callback function.
      // If the function name should not appear at the beginning of the error message then set
      // the flag emNoFuncNames (pdf.SetErrorMode(DynaPDF.TErrMode.emNoFuncNames);).
      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write("{0}\n", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage));
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

		[STAThread]
		static void Main(string[] args)
		{
         try
         {
            double x, y, w, h, pw, ph, incX, incY;
            int i, nx, ny, xx, yy, cnt = TEST_CODES.Length;
            
            CPDF pdf = new CPDF();
            // Error messages and warnings are passed to the callback function.
            pdf.SetOnErrorProc(IntPtr.Zero, new DynaPDF.TErrorProc(PDFError));
            pdf.CreateNewPDF(null); // We open the output file later if no error occurrs.

            pdf.SetPageCoords(TPageCoord.pcTopDown);

            TPDFBarcode2 bcd = new TPDFBarcode2();
            pdf.InitBarcode2(ref bcd);

            bcd.Options = TPDFBarcodeOptions.bcoDefault | TPDFBarcodeOptions.bcoUseActiveFont;
            // Use this flag if you want to create image based barcodes
            //bcd.Options = DynaPDF.TPDFBarcodeOptions.bcoImageOutput;

            pw   = pdf.GetPageWidth()  - 100.0;
            ph   = pdf.GetPageHeight() - 100.0;
            w    = 100.0;
            h    = 120.0;
            nx   = (int)(pw / w);
            ny   = (int)(ph / h);
            incX = w + (pw - nx * w) / (nx-1);
            incY = h + (ph - ny * h) / (ny-1);
            h    = 100.0;

            for (i = 0; i < cnt; )
            {
               pdf.Append();

               pdf.SetFont("Helvetica", TFStyle.fsRegular, 6.5, true, TCodepage.cp1252);
               pdf.SetLineWidth(0.0);

               y = 50.0;
               for (yy = 0; yy < ny && i < cnt; yy++, y += incY)
               {
                  for (xx = 0, x = 50.0; xx < nx && i < cnt; xx++, i++, x += incX)
                  {
                     TTestBarcode b = TEST_CODES[i];
                     bcd.BarcodeType = b.BarcodeType;
                     bcd.Data        = b.Data;
                     bcd.DataType    = b.DataType;
                     bcd.Primary     = b.Primary;

                     pdf.WriteFTextEx(x, y-10.0, w, -1.0, TTextAlign.taCenter, b.BarcodeName);
                     pdf.Rectangle(x, y, w, h, TPathFillMode.fmStroke);
                     pdf.InsertBarcode(x, y, w, h, TCellAlign.coCenter, TCellAlign.coCenter, ref bcd);
                  }
               }
               pdf.EndPage();
            }
            // No fatal error occurred?
            if (pdf.HaveOpenDoc())
            {
               // We write the output file into the current directory.
               String filePath = Directory.GetCurrentDirectory() + "\\out.pdf";
               // OK, now we can open the output file.
               if (!pdf.OpenOutputFile(filePath)) return;
               if (pdf.CloseFile())
               {
                  Console.Write("PDF file \"{0}\" successfully created!\n", filePath);
                  System.Diagnostics.Process p = new System.Diagnostics.Process();
                  p.StartInfo.FileName = filePath;
                  p.Start();
               }
            }
            pdf = null;
         }catch(Exception e)
         {
            Console.Write(e.Message);
         }
         Console.Read();
		}
	}
}
